import { __rest } from "tslib";
import QRCode from 'qrcode';
import { defineComponent, h, ref, watch } from 'vue-demi';
export const LEVELS = [
    'low',
    'medium',
    'quartile',
    'high',
    'L',
    'M',
    'Q',
    'H',
];
// eslint-disable-next-line @typescript-eslint/no-magic-numbers
export const MASK_PATTERNS = [0, 1, 2, 3, 4, 5, 6, 7];
export const MODES = ['alphanumeric', 'numeric', 'kanji', 'byte'];
export const TYPES = ['image/png', 'image/jpeg', 'image/webp'];
const MAX_QR_VERSION = 40;
export default defineComponent({
    props: {
        version: {
            type: Number,
            validator: (version) => version === Number.parseInt(String(version), 10) &&
                version >= 1 &&
                version <= MAX_QR_VERSION,
        },
        errorCorrectionLevel: {
            type: String,
            validator: (level) => LEVELS.includes(level),
        },
        maskPattern: {
            type: Number,
            validator: (maskPattern) => MASK_PATTERNS.includes(maskPattern),
        },
        toSJISFunc: Function,
        margin: Number,
        scale: Number,
        width: Number,
        color: {
            type: Object,
            validator: (color) => ['dark', 'light'].every(c => ['string', 'undefined'].includes(typeof color[c])),
            required: true,
        },
        type: {
            type: String,
            validator: (type) => TYPES.includes(type),
            required: true,
        },
        quality: {
            type: Number,
            validator: (quality) => quality === Number.parseFloat(String(quality)) &&
                quality >= 0 &&
                quality <= 1,
            required: true,
        },
        value: {
            type: [String, Array],
            required: true,
            validator(value) {
                if (typeof value === 'string') {
                    return true;
                }
                return value.every(({ data, mode }) => typeof data === 'string' && MODES.includes(mode));
            },
        },
    },
    setup(props, { attrs, emit }) {
        const dataUrlRef = ref();
        const toDataURL = () => {
            // no idea why, but vuepress emits error saying `props` is undefined
            // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
            const _a = props || {}, { quality, value } = _a, rest = __rest(_a, ["quality", "value"]);
            QRCode.toDataURL(value, Object.assign(rest, quality == null || {
                renderOptions: {
                    quality,
                },
            }))
                .then(dataUrl => {
                dataUrlRef.value = dataUrl;
            })
                .catch((err) => emit('error', err));
        };
        watch(props, toDataURL, { immediate: true });
        return () => h('img', Object.assign(Object.assign({}, attrs), { src: dataUrlRef.value }));
    },
});
//# sourceMappingURL=index.js.map