"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Transaction = exports.OPERATION_TYPE = exports.ISOLATION_LEVEL = void 0;
exports.assertValidIsolationLevel = assertValidIsolationLevel;
exports.isolationLevelByValue = void 0;
var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));
var _allHeaders = require("./all-headers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
  s2.2.6.8
 */

const OPERATION_TYPE = exports.OPERATION_TYPE = {
  TM_GET_DTC_ADDRESS: 0x00,
  TM_PROPAGATE_XACT: 0x01,
  TM_BEGIN_XACT: 0x05,
  TM_PROMOTE_XACT: 0x06,
  TM_COMMIT_XACT: 0x07,
  TM_ROLLBACK_XACT: 0x08,
  TM_SAVE_XACT: 0x09
};
const ISOLATION_LEVEL = exports.ISOLATION_LEVEL = {
  NO_CHANGE: 0x00,
  READ_UNCOMMITTED: 0x01,
  READ_COMMITTED: 0x02,
  REPEATABLE_READ: 0x03,
  SERIALIZABLE: 0x04,
  SNAPSHOT: 0x05
};
const isolationLevelByValue = exports.isolationLevelByValue = {};
for (const name in ISOLATION_LEVEL) {
  const value = ISOLATION_LEVEL[name];
  isolationLevelByValue[value] = name;
}
function assertValidIsolationLevel(isolationLevel, name) {
  if (typeof isolationLevel !== 'number') {
    throw new TypeError(`The "${name}" ${name.includes('.') ? 'property' : 'argument'} must be of type number. Received type ${typeof isolationLevel} (${isolationLevel})`);
  }
  if (!Number.isInteger(isolationLevel)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be an integer. Received: ${isolationLevel}`);
  }
  if (!(isolationLevel >= 0 && isolationLevel <= 5)) {
    throw new RangeError(`The value of "${name}" is out of range. It must be >= 0 && <= 5. Received: ${isolationLevel}`);
  }
}
class Transaction {
  constructor(name, isolationLevel = ISOLATION_LEVEL.NO_CHANGE) {
    this.name = name;
    this.isolationLevel = isolationLevel;
    this.outstandingRequestCount = 1;
  }
  beginPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ucs2');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_BEGIN_XACT);
    buffer.writeUInt8(this.isolationLevel);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },
      toString: () => {
        return 'Begin Transaction: name=' + this.name + ', isolationLevel=' + isolationLevelByValue[this.isolationLevel];
      }
    };
  }
  commitPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_COMMIT_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    // No fBeginXact flag, so no new transaction is started.
    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },
      toString: () => {
        return 'Commit Transaction: name=' + this.name;
      }
    };
  }
  rollbackPayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_ROLLBACK_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    // No fBeginXact flag, so no new transaction is started.
    buffer.writeUInt8(0);
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },
      toString: () => {
        return 'Rollback Transaction: name=' + this.name;
      }
    };
  }
  savePayload(txnDescriptor) {
    const buffer = new _writableTrackingBuffer.default(100, 'ascii');
    (0, _allHeaders.writeToTrackingBuffer)(buffer, txnDescriptor, this.outstandingRequestCount);
    buffer.writeUShort(OPERATION_TYPE.TM_SAVE_XACT);
    buffer.writeUInt8(this.name.length * 2);
    buffer.writeString(this.name, 'ucs2');
    return {
      *[Symbol.iterator]() {
        yield buffer.data;
      },
      toString: () => {
        return 'Save Transaction: name=' + this.name;
      }
    };
  }
  isolationLevelToTSQL() {
    switch (this.isolationLevel) {
      case ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'READ UNCOMMITTED';
      case ISOLATION_LEVEL.READ_COMMITTED:
        return 'READ COMMITTED';
      case ISOLATION_LEVEL.REPEATABLE_READ:
        return 'REPEATABLE READ';
      case ISOLATION_LEVEL.SERIALIZABLE:
        return 'SERIALIZABLE';
      case ISOLATION_LEVEL.SNAPSHOT:
        return 'SNAPSHOT';
    }
    return '';
  }
}
exports.Transaction = Transaction;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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